////////////////////////////////////////////////////////////////////////////////
//                                                                             /
// 2012-2020 (c) Baical                                                        /
//                                                                             /
// This library is free software; you can redistribute it and/or               /
// modify it under the terms of the GNU Lesser General Public                  /
// License as published by the Free Software Foundation; either                /
// version 3.0 of the License, or (at your option) any later version.          /
//                                                                             /
// This library is distributed in the hope that it will be useful,             /
// but WITHOUT ANY WARRANTY; without even the implied warranty of              /
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           /
// Lesser General Public License for more details.                             /
//                                                                             /
// You should have received a copy of the GNU Lesser General Public            /
// License along with this library.                                            /
//                                                                             /
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//                    GOST 28147-89 crypt. algorithm                          //
////////////////////////////////////////////////////////////////////////////////
#ifndef GOST2814789_H
#define GOST2814789_H


#define GOST_PASSWORD_LENGTH                                               (32)
#define GOST_EXCH_TABLE_WIDTH                                              (8)
#define GOST_EXCH_TABLE_HEIGHT                                             (16)
#define GOST_SEQUENCE_LENGTH                                               (32)
#define GOST_K_EMPTY                                                       (128)

#if defined(_WIN32) || defined(_WIN64)
#else
    #define _rotl(value, shift) (((value) << (shift)) | ((value) >> (32 - shift)))
#endif


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//                           Base abstract class
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
class CGOST2814789
{
public:
    enum eMode
    {
        EMODE_CBC,
        EMODE_CBF,
        EMODE_ESN,
        EMODE_RND
    };

protected:
   tUINT8        m_pK[GOST_EXCH_TABLE_WIDTH][GOST_EXCH_TABLE_HEIGHT]; 
   tUINT32       m_pSeq[GOST_SEQUENCE_LENGTH];
   tUINT32       m_pX[GOST_PASSWORD_LENGTH / 4];//System key
   tUINT32       m_pN1[2];                   //Start value to main cycle                               
   tUINT32       m_dwN3;
   tUINT32       m_dwN4;                     //Memory for gamming algorithm
   const tUINT32 m_dwN5;
   const tUINT32 m_dwN6;
   eMode         m_eMode;
   tBOOL         m_bInitialized;
   tBOOL         m_bCompativilityMode;
public:
    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789::CGOST2814789
    CGOST2814789(eMode i_eMode)
         : m_dwN3(0)
         , m_dwN4(0)
         , m_dwN5(0x1010101) //C2
         , m_dwN6(0x1010104) //C1
         , m_eMode(i_eMode)
         , m_bInitialized(FALSE)
         , m_bCompativilityMode(TRUE)
    {
        const tUINT8 l_pK[GOST_EXCH_TABLE_WIDTH][GOST_EXCH_TABLE_HEIGHT] =
        {
            {0xC, 0x4, 0x6, 0x2, 0xA, 0x5, 0xB, 0x9, 0xE, 0x8, 0xD, 0x7, 0x0, 0x3, 0xF, 0x1},
            {0x6, 0x8, 0x2, 0x3, 0x9, 0xA, 0x5, 0xC, 0x1, 0xE, 0x4, 0x7, 0xB, 0xD, 0x0, 0xF},
            {0xB, 0x3, 0x5, 0x8, 0x2, 0xF, 0xA, 0xD, 0xE, 0x1, 0x7, 0x4, 0xC, 0x9, 0x6, 0x0},
            {0xC, 0x8, 0x2, 0x1, 0xD, 0x4, 0xF, 0x6, 0x7, 0x0, 0xA, 0x5, 0x3, 0xE, 0x9, 0xB},
            {0x7, 0xF, 0x5, 0xA, 0x8, 0x1, 0x6, 0xD, 0x0, 0x9, 0x3, 0xE, 0xB, 0x4, 0x2, 0xC},
            {0x5, 0xD, 0xF, 0x6, 0x9, 0x2, 0xC, 0xA, 0xB, 0x7, 0x8, 0x1, 0x4, 0x3, 0xE, 0x0},
            {0x8, 0xE, 0x2, 0x5, 0x6, 0x9, 0x1, 0xC, 0xF, 0x4, 0xB, 0x0, 0xD, 0xA, 0x3, 0x7},
            {0x1, 0x7, 0xE, 0xD, 0x0, 0x5, 0x8, 0x3, 0x4, 0xF, 0xA, 0x6, 0x9, 0xC, 0xB, 0x2}

            // {10,14, 9, 3,15, 8,11,12, 5, 2, 0, 6, 7,13, 1, 4},
            // { 8, 4,12, 9, 1,10,14,11, 0, 3, 5, 7, 2,13, 6,15},
            // { 8,12,10, 9,14,15, 6,13, 0, 3, 2,11, 1, 7, 4, 5},
            // {15,12, 7,14, 6,13, 4, 2, 9, 8,11,10, 1, 5, 3, 0},
            // {11, 5,13, 8,14, 1, 6,12, 3,15,10, 0, 7, 2, 4, 9},
            // { 5,13, 8,12, 9, 0, 6,15, 2, 4,11,10, 3, 1,14, 7},
            // { 0, 1,11, 6,14,10, 3,12, 8, 9, 5, 2, 7,15, 4,13},
            // { 0,12, 7, 5, 9, 3,14, 2,10, 4, 8,13, 1,11, 6,15}
        };
    
        //Sequence of main manipulation with 2x32bits blocks
        const tUINT32 l_pSeq[GOST_SEQUENCE_LENGTH]=
        {
            0,1,2,3,4,5,6,7,
            0,1,2,3,4,5,6,7,
            0,1,2,3,4,5,6,7,
            7,6,5,4,3,2,1,0
        };

   
        memcpy(m_pK,   l_pK,   sizeof(l_pK));
        memcpy(m_pSeq, l_pSeq, sizeof(l_pSeq));

        memset(m_pX, 0, sizeof(m_pX));
        memset(m_pN1, 0, sizeof(m_pN1));
    }//CGOST2814789::CGOST2814789


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789::~CGOST2814789
    ~CGOST2814789()
    {
    
    }//CGOST2814789::~CGOST2814789

    ////////////////////////////////////////////////////////////////////////////
    void SetCompatibilityMode(tBOOL i_bOn)
    {
        m_bCompativilityMode = i_bOn;
    }

    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789::Set_Password
    // User can specify exchange table, it should have size
    // [GOST_EXCH_TABLE_WIDTH][GOST_EXCH_TABLE_HEIGHT]
    void Set_Password(const tUINT8  i_pPassword[GOST_PASSWORD_LENGTH], 
                      const tUINT8 *i_pK = NULL
                     )
    {
        if (m_bCompativilityMode)
        {
            memcpy(m_pX, i_pPassword, sizeof(tUINT32));
        }
        else
        {
            memcpy(m_pX, i_pPassword, GOST_PASSWORD_LENGTH);
        }

        if (i_pK)
        {
            memcpy(m_pK, i_pK, sizeof(m_pK));
        }

        m_bInitialized = TRUE;
    }//CGOST2814789::Set_Password


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789::Set_Synchro
    // This function should be called if new logical encryption or decryption
    // will be started
    virtual void Set_Synchro(const tUINT32 i_pSynchro[2])                   = 0;


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789::Base_Encrypt
    //To the input - 64 bit as 2 uint32
    void Encrypt64(tUINT32 io_pQWord[2])
    { 
        tUINT32  l_pExch = 0;
        //Pointer to 32 bits block and translate him to tUINT8 form
        tUINT8 * l_pN    = (tUINT8 *)&io_pQWord[0]; 

        for (tUINT32 l_dwI=0; l_dwI < 32; l_dwI++)
        {
            l_pExch      = io_pQWord[0];                                                                                                                                          
            io_pQWord[0] = (tUINT32)(io_pQWord[0] + m_pX[m_pSeq[l_dwI]]);

            //Exchange 32 bits from table
            //l_pN[0]&0xF - set on 4 fist bits - value [0..15]
            //((l_pN[0]&0xF0)>>4) - set on last 4  bits - value [0..15]
            // l_pQWord[0]=[7,6,5,4,3,2,1,0] where each ceil is 4 bits value and 
            // all ceil changed from table.
            // Set on 8 bits - 2 ceil 4 bits 1,2 ceil
            l_pN[0] = m_pK[0][(l_pN[0]&0xF)] | ((m_pK[1][(l_pN[0]&0xF0)>>4])<<4); 
            l_pN[1] = m_pK[2][(l_pN[1]&0xF)] | ((m_pK[3][(l_pN[1]&0xF0)>>4])<<4); 
            l_pN[2] = m_pK[4][(l_pN[2]&0xF)] | ((m_pK[5][(l_pN[2]&0xF0)>>4])<<4); 
            l_pN[3] = m_pK[6][(l_pN[3]&0xF)] | ((m_pK[7][(l_pN[3]&0xF0)>>4])<<4); 

            io_pQWord[0] = _rotl(io_pQWord[0], 11) ^ io_pQWord[1];
            io_pQWord[1] = l_pExch;
        }
        io_pQWord[1] = io_pQWord[0];
        io_pQWord[0] = l_pExch;
    }//CGOST2814789::Base_Encrypt


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789::Base_Decrypt
    //To the input - 64 bit as 2 uint32
    void Decrypt64(tUINT32 io_pQWord[2])
    { 
        tUINT32  l_pExch = 0;
        //Pointer to 32 bits block and translate him to tUINT8 form
        tUINT8 * l_pN    = (tUINT8 *)&io_pQWord[0]; 

        for (tUINT32 l_dwI=0; l_dwI < 32; l_dwI++)
        {
            l_pExch      = io_pQWord[0];                                                                                                                                          
            io_pQWord[0] = (tUINT32)(io_pQWord[0] + m_pX[m_pSeq[31 - l_dwI]]);

            //Exchange 32 bits from table
            //l_pN[0]&0xF - set on 4 fist bits - value [0..15]
            //((l_pN[0]&0xF0)>>4) - set on last 4  bits - value [0..15]
            // l_pQWord[0]=[7,6,5,4,3,2,1,0] where each ceil is 4 bits value and 
            // all ceil changed from table.
            // Set on 8 bits - 2 ceil 4 bits 1,2 ceil
            l_pN[0] = m_pK[0][(l_pN[0]&0xF)] | ((m_pK[1][(l_pN[0]&0xF0)>>4])<<4);
            l_pN[1] = m_pK[2][(l_pN[1]&0xF)] | ((m_pK[3][(l_pN[1]&0xF0)>>4])<<4);
            l_pN[2] = m_pK[4][(l_pN[2]&0xF)] | ((m_pK[5][(l_pN[2]&0xF0)>>4])<<4);
            l_pN[3] = m_pK[6][(l_pN[3]&0xF)] | ((m_pK[7][(l_pN[3]&0xF0)>>4])<<4);

            io_pQWord[0] = _rotl(io_pQWord[0], 11) ^ io_pQWord[1];
            io_pQWord[1] = l_pExch;
        }
        io_pQWord[1] = io_pQWord[0];
        io_pQWord[0] = l_pExch;
    }//CGOST2814789::Base_Decrypt   
}; //CGOST2814789



////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//                     Cipher Block Chaining (CBC)
// Simple block encryption, cipher generate random gamma using syncro-vector &
// XOR this gamma with open data, encrypted 64 bits blocks are independent,
// if encrypted data (1byte) will be changed - in decrypted data only one 64 bit
// block will be damaged.
// Usage:
// - Set_Password() - once
// - For every independent logical block
//   - Set_Synchro()
//   - Crypt()
// N.B.: do not recommended to use this cipher for plain text data, only for
//       random values like keys. 
//       Synchro-vector should be random for every logical encryp/decrypt.
//       Not safe for changing values inside encrypted data
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
class CGOST2814789_CBC
    : public CGOST2814789
{
public:
    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CBC::CGOST2814789_CBC
    CGOST2814789_CBC()
        : CGOST2814789(CGOST2814789::EMODE_CBC)
    {
    }//CGOST2814789_CBC::CGOST2814789_CBC


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CBC::~CGOST2814789_CBC
    ~CGOST2814789_CBC()
    {

    }//CGOST2814789_CBC::~CGOST2814789_CBC


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CBC::Set_Synchro
    virtual void Set_Synchro(const tUINT32 i_pSynchro[2])
    {
        //Create synchro point;
        m_pN1[0]   = i_pSynchro[0]; 
        m_pN1[1]   = i_pSynchro[1];
    
        Encrypt64(&m_pN1[0]);
    
        m_dwN3 = m_pN1[0];
        m_dwN4 = m_pN1[1];
    }//CGOST2814789_CBC::Set_Synchro


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CBC::Crypt
    // Before using this function you should call
    // - Set_Password(): once
    // - Set_Synchro() : for every logical independent data block before enc/dec
    //                   synchro data should be different !  
    // Parameters:
    // o_pBuffer  - pointer to 32bits words
    // i_szLength - count of 32bits words, should be even (i_szLength%2 = 0)
    //
    // N.B.: Function is used for encryption & decryption !
    tBOOL Crypt(tUINT32 *i_pBuffer, size_t i_szLength)
    {
        tUINT64 l_qwCM;
        if (    (NULL  == i_pBuffer)
             || (0     == i_szLength)
             || (FALSE == m_bInitialized)
             || (i_szLength & 1)
           )
        {
            return FALSE;
        }

        i_szLength /= 2;

        for (tUINT32 l_dwI = 0; l_dwI < i_szLength; l_dwI++)
        {   //Create gamming code
            m_dwN3   = (tUINT32)(m_dwN3 + m_dwN5); 

            l_qwCM   = (tUINT64)m_dwN4 + (tUINT64)m_dwN6;

            if (l_qwCM >= 0x100000000ULL)
            {
                l_qwCM = l_qwCM - 0xFFFFFFFFULL;
            }

            m_dwN4   = (tUINT32)l_qwCM;


            m_pN1[0] = m_dwN3; //Second step
            m_pN1[1] = m_dwN4;

            Encrypt64(&m_pN1[0]);  //Third step

            //Put gamma on open data
            (*i_pBuffer)  ^= m_pN1[0];        
            i_pBuffer ++;
            *i_pBuffer    ^= m_pN1[1];    
            i_pBuffer ++;
        }

        return TRUE;
    }//CGOST2814789_CBC::Crypt
}; //CGOST2814789_CBC




////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//                        Cipher Feedback (CFB)
// Strong block encryption, cipher generate random gamma using syncro-vector &
// XOR this gamma with open data, then reuse encrypted data for next encryption
// iteration. This cipher is used to encrypt text & other non random or other
// repeating data
// 
// Usage:
// - Set_Password() - once
// - For every independent logical block
//   - Set_Synchro()
//   - Encrypt()/Decrypt()
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
class CGOST2814789_CFB
    : public CGOST2814789
{
public:
    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CFB::CGOST2814789_CFB
    CGOST2814789_CFB()
        : CGOST2814789(CGOST2814789::EMODE_CBF)
    {
    }//CGOST2814789_CFB::CGOST2814789_CFB


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CFB::~CGOST2814789_CFB
    ~CGOST2814789_CFB()
    {
    }//CGOST2814789_CFB::~CGOST2814789_CFB


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CFB::Set_Synchro
    virtual void Set_Synchro(const tUINT32 i_pSynchro[2])
    {
        m_pN1[0]   = i_pSynchro[0]; 
        m_pN1[1]   = i_pSynchro[1];
    }////CGOST2814789_CFB::Set_Synchro


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CFB::Encrypt
    // Before using this function you should call
    // - Set_Password(): once
    // - Set_Synchro() : for every logical independent data block before enc/dec
    //                   synchro data should be different !  
    // Parameters:
    // o_pBuffer  - pointer to 32bits words
    // i_szLength - count of 32bits words, should be even (i_szLength%2 = 0)
    tBOOL Encrypt(tUINT32 *i_pBuffer, size_t i_szLength)
    {
        if (    (NULL  == i_pBuffer)
             || (0     == i_szLength)
             || (FALSE == m_bInitialized)
             || (i_szLength & 1)
           )
        {
            return FALSE;
        }

        i_szLength /= 2;

        for (tUINT32 l_dwI = 0; l_dwI < i_szLength; l_dwI++)
        {
            Encrypt64(&m_pN1[0]);  
            //Put gamma on open data
            (*i_pBuffer) ^= m_pN1[0];        
            m_pN1[0]      = (*i_pBuffer);        
            i_pBuffer ++;

            (*i_pBuffer) ^= m_pN1[1];    
            m_pN1[1]      = (*i_pBuffer);    
            i_pBuffer ++;
        }

        return TRUE;
    }//CGOST2814789_CFB::Crypt


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CFB::Decrypt
    // Before using this function you should call
    // - Set_Password(): once
    // - Set_Synchro() : for every logical independent data block before enc/dec
    //                   synchro data should be different !  
    // Parameters:
    // o_pBuffer  - pointer to 32bits words
    // i_szLength - count of 32bits words, should be even (i_szLength%2 = 0)
    tBOOL Decrypt(tUINT32 *i_pBuffer, size_t i_szLength)
    {
        tUINT32 l_pExch;

        if (    (NULL  == i_pBuffer)
             || (0     == i_szLength)
             || (FALSE == m_bInitialized)
             || (i_szLength & 1)
           )
        {
            return FALSE;
        }

        i_szLength /= 2;

        for (tUINT32 l_dwI = 0; l_dwI < i_szLength; l_dwI++)
        {
            Encrypt64(&m_pN1[0]);  
            //Put gamma on open data
            l_pExch       = (*i_pBuffer);
            (*i_pBuffer) ^= m_pN1[0];        
            m_pN1[0]      = l_pExch;        
            i_pBuffer ++;

            l_pExch       = (*i_pBuffer);
            (*i_pBuffer) ^= m_pN1[1];    
            m_pN1[1]      = l_pExch;    
            i_pBuffer ++;
        }

        return TRUE;
    }//CGOST2814789_CFB::Decrypt

}; //CGOST2814789_CFB



////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//                         Electronic signature
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
class CGOST2814789_ESN
    : public CGOST2814789
{
protected:
   tUINT32 m_pN1_ES[2]; //Start value to electronic signature

public:
    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_ESN::CGOST2814789_ESN
    CGOST2814789_ESN()
        : CGOST2814789(CGOST2814789::EMODE_ESN)
    {
    }//CGOST2814789_ESN::CGOST2814789_ESN


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_ESN::~CGOST2814789_ESN
    ~CGOST2814789_ESN()
    {
    }//CGOST2814789_ESN::~CGOST2814789_ESN


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_ESN::Set_Synchro
    virtual void Set_Synchro(const tUINT32 i_pSynchro[2])
    {
        m_pN1_ES[0] = i_pSynchro[0]; //For Electronic signature
        m_pN1_ES[1] = i_pSynchro[1];
    }//CGOST2814789_ESN::Set_Synchro


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_ESN::Update
    // Before using this function you should call
    // - Set_Password(): once
    // - Set_Synchro() : for every logical independent data block before signing
    // Parameters:
    // o_pBuffer  - pointer to 32bits words
    // i_szLength - count of 32bits words, should be even (i_szLength%2 = 0)
    tBOOL Update(const tUINT32 *i_pBuffer, size_t i_szLength)
    { 
        if (    (NULL  == i_pBuffer)
             || (0     == i_szLength)
             || (FALSE == m_bInitialized)
             || (i_szLength & 1)
           )
        {
            return FALSE;
        }

        i_szLength /= 2;

        for (tUINT32 l_dwI = 0; l_dwI < i_szLength; l_dwI++ )
        { 
            //16 cycles with synchropost 
            Sign64(&m_pN1_ES[0]);

            //Put mask of base data on synchropost  - Xor
            m_pN1_ES[0] ^= *i_pBuffer;
            i_pBuffer ++;

            m_pN1_ES[1] ^= *i_pBuffer;
            i_pBuffer ++;
        }

        return TRUE;
    }//CGOST2814789_ESN::Update


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_ESN::Get
    tUINT64 Get()
    {
        return *(tUINT64*)m_pN1_ES;
    }//CGOST2814789_ESN::Get


protected:
    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_ESN::Base_Sign
    void Sign64(tUINT32 io_pQWord[2])
    {
        tUINT32  l_pExch = 0;
        //Pointer to 32 bits block and translate him to tUINT8 form
        tUINT8  *l_pN    = (tUINT8 *)&io_pQWord[0]; 

        for (tUINT32 l_dwI = 0; l_dwI < 16; l_dwI++)
        {
            l_pExch = io_pQWord[0];
            io_pQWord[0] = (tUINT32)(io_pQWord[0] + m_pX[m_pSeq[l_dwI]]);

            //Exchange 32 bits from table
            //l_pN[0]&0xF - set on 4 fist bits - value [0..15]
            //((l_pN[0]&0xF0)>>4) - set on last 4  bits - value [0..15]
            // l_pQWord[0]=[7,6,5,4,3,2,1,0] where each ceil is 4 bits value and 
            // all ceil changed from table.
            //Set on 8 bits - 2 ceil 4 bits 1,2 ceil
            l_pN[0] = m_pK[0][(l_pN[0]&0xF)] | ((m_pK[1][(l_pN[0]&0xF0)>>4])<<4);
            l_pN[1] = m_pK[2][(l_pN[1]&0xF)] | ((m_pK[3][(l_pN[1]&0xF0)>>4])<<4); 
            l_pN[2] = m_pK[4][(l_pN[2]&0xF)] | ((m_pK[5][(l_pN[2]&0xF0)>>4])<<4); 
            l_pN[3] = m_pK[6][(l_pN[3]&0xF)] | ((m_pK[7][(l_pN[3]&0xF0)>>4])<<4); 

            io_pQWord[0] = _rotl(io_pQWord[0], 11) ^ io_pQWord[1];

            io_pQWord[1] = l_pExch;
        }
    }//CGOST2814789_ESN::Base_Sign

};//CGOST2814789_ESN



////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//                        Gamma (random) generation
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
class CGOST2814789_RND
    : public CGOST2814789
{
public:
    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_RND::CGOST2814789_RND
    CGOST2814789_RND()
        : CGOST2814789(CGOST2814789::EMODE_RND)
    {
    }//CGOST2814789_RND::CGOST2814789_RND


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_RND::~CGOST2814789_RND
    ~CGOST2814789_RND()
    {
    }//CGOST2814789_RND::~CGOST2814789_RND


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_CBC::Set_Synchro
    virtual void Set_Synchro(const tUINT32 i_pSynchro[2])
    {
        //Create synchro point;
        m_pN1[0]   = i_pSynchro[0]; 
        m_pN1[1]   = i_pSynchro[1];
    
        Encrypt64(&m_pN1[0]);
    
        m_dwN3 = m_pN1[0];
        m_dwN4 = m_pN1[1];
    }////CGOST2814789_CBC::Set_Synchro


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_RND::Get_Qword
    // Before using this function you should call
    // - Set_Password(): once
    // - Set_Synchro() : once
    tUINT64 Get_Qword()
    {
        tUINT64 l_qwCM;

        if (FALSE == m_bInitialized)
        {
            return FALSE;
        }

        m_dwN3   = (tUINT32)(m_dwN3 + m_dwN5); 
        l_qwCM   = (tUINT64)m_dwN4 + (tUINT64)m_dwN6;

        if (l_qwCM >= 0x100000000ULL)
        {
            l_qwCM = l_qwCM - 0xFFFFFFFFULL;
        }

        m_dwN4   = (tUINT32)l_qwCM;

        m_pN1[0] = m_dwN3; //Second step
        m_pN1[1] = m_dwN4;

        Encrypt64(&m_pN1[0]);  //Third step

        return *(tUINT64*)m_pN1;
    }//CGOST2814789_RND::Get_Qword


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_RND::Get_Array
    // Before using this function you should call
    // - Set_Password(): once
    // - Set_Synchro() : once
    // Parameters:
    // o_pBuffer  - pointer to 8bits array
    // i_szLength - count of 8bits elements
    tBOOL Get_Array(tUINT8 *o_pBuffer, size_t i_szLength)
    {
        tUINT64 l_qwCM;
        tUINT32 *l_pBuffer    = (tUINT32 *)o_pBuffer;
        size_t   l_szQWords   = i_szLength / 8;
        size_t   l_szReminder = i_szLength % 8;

        if (    (NULL  == o_pBuffer)
             || (0     == i_szLength)
             || (FALSE == m_bInitialized)
             || (0     >= i_szLength)
           )
        {
            return FALSE;
        }

        for (tUINT32 l_dwI = 0; l_dwI < l_szQWords; l_dwI++)
        {   //Create gamming code
            m_dwN3   = (tUINT32)(m_dwN3 + m_dwN5); 
            l_qwCM   = (tUINT64)m_dwN4 + (tUINT64)m_dwN6;

            if (l_qwCM >= 0x100000000ULL)
            {
                l_qwCM = l_qwCM - 0xFFFFFFFFULL;
            }

            m_dwN4   = (tUINT32)l_qwCM;

            m_pN1[0] = m_dwN3; //Second step
            m_pN1[1] = m_dwN4;

            Encrypt64(&m_pN1[0]);  //Third step

            //Put gamma on open data
            (*l_pBuffer) = m_pN1[0];        
            l_pBuffer ++;
            *l_pBuffer   = m_pN1[1];    
            l_pBuffer ++;
        }

        if (l_szReminder) 
        {
            tUINT64 l_qwPadding = Get_Qword();

            o_pBuffer += l_szQWords * 8;

            while (l_szReminder)
            {
                *o_pBuffer = (tUINT8)l_qwPadding;
                l_qwPadding = l_qwPadding >> 8;
                o_pBuffer ++;
                --l_szReminder;
            }
        }

        return TRUE;
    }//CGOST2814789_RND::Get_Array


    ////////////////////////////////////////////////////////////////////////////
    //CGOST2814789_RND::Generate_K
    const tUINT8 *Generate_K()
    { 
        tUINT8 l_bR = 0;
        tUINT8 l_bJ = 0;
        tUINT8 l_bI = 0;


        memset(m_pK, GOST_K_EMPTY, sizeof(m_pK));

        for (l_bJ = 0; l_bJ < 8; l_bJ++)
        {
            for (l_bI = 0; l_bI < 16; l_bI++)
            {
                while (    (GOST_K_EMPTY != m_pK[l_bJ][l_bI])
                        && (l_bI < 16)
                      )
                {
                    l_bI++; //If element not empty
                }

                if (l_bI > 15) 
                {
                    break; //if it was last element
                }
    
                do 
                {
                    l_bR=(tUINT8)(Get_Qword() & 0xF);//Generate one tUINT8
                } while (/*(0 <= m_pK[l_bJ][l_bR]) ||*/ (m_pK[l_bJ][l_bI] == l_bR));
             
                m_pK[l_bJ][l_bI]=l_bR; //Put data to table
                m_pK[l_bJ][l_bR]=l_bI;
            }
        }

        return (tUINT8*)m_pK;
    }//CGOST2814789_RND::Generate_K
}; //CGOST2814789_RND

#endif //GOST2814789_H