////////////////////////////////////////////////////////////////////////////////
//                                                                             /
// 2012-2021 (c) Baical                                                        /
//                                                                             /
// This library is free software; you can redistribute it and/or               /
// modify it under the terms of the GNU Lesser General Public                  /
// License as published by the Free Software Foundation; either                /
// version 3.0 of the License, or (at your option) any later version.          /
//                                                                             /
// This library is distributed in the hope that it will be useful,             /
// but WITHOUT ANY WARRANTY; without even the implied warranty of              /
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           /
// Lesser General Public License for more details.                             /
//                                                                             /
// You should have received a copy of the GNU Lesser General Public            /
// License along with this library.                                            /
//                                                                             /
////////////////////////////////////////////////////////////////////////////////
#include "uP7preCommon.h"

static const tXCHAR *g_puP7files[] = 
{
    TM("uP7.c"),
    TM("uP7.h"),
    TM("uP7preprocessed.h")
};



////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
CpreManager::CpreManager()
    : m_pSrcDir(NULL)
    , m_pOutDir(NULL)
    , m_pName(NULL)
    , m_eError(eErrorNo)
    , m_szTargetCpuBits(32)
{

}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
CpreManager::~CpreManager()
{
    m_cFunctions.Clear(TRUE);
    m_cFiles.Clear(TRUE);

    if (m_pSrcDir)
    {
        PStrFreeDub(m_pSrcDir);
        m_pSrcDir = NULL;
    }

    if (m_pOutDir)
    {
        PStrFreeDub(m_pOutDir);
        m_pOutDir = NULL;
    }

    if (m_pName)
    {
        PStrFreeDub(m_pName);
        m_pName = NULL;
    }
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void CpreManager::SetSourcesDir(const tXCHAR* i_pDir)
{
    if (m_pSrcDir)
    {
        PStrFreeDub(m_pSrcDir);
    }

    m_pSrcDir = NormalizePath(i_pDir);
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void CpreManager::SetOutputDir(const tXCHAR* i_pDir)
{
    if (m_pOutDir)
    {
        PStrFreeDub(m_pOutDir);
    }

    m_pOutDir = NormalizePath(i_pDir);
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void CpreManager::SetTargetCpuBitsCount(size_t i_szBits)
{
    m_szTargetCpuBits = i_szBits;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void CpreManager::SetName(const tXCHAR *i_pName)
{
    if (m_pName)
    {
        PStrFreeDub(m_pName);
    }

    m_pName = NormalizePath(i_pName);
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void CpreManager::SetTargetCpuWCharBitsCount(size_t i_szBits)
{
    m_szWcharBits = i_szBits;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
eErrorCodes CpreManager::AddFile(const tXCHAR *i_pPath)
{
    size_t l_szNewFile = PStrLen(i_pPath);
    bool   l_bAdd      = true;

    for (size_t l_szI = 0; l_szI < LENGTH(g_puP7files); l_szI++)
    {
        size_t l_szuP7File = PStrLen(g_puP7files[l_szI]);
        if (    (l_szNewFile > l_szuP7File)
             && (0 == PStrCmp(i_pPath + l_szNewFile - l_szuP7File, g_puP7files[l_szI]))
           )
        {
            //skip file - it is part of up7, we do need to scan it
            //OSPRINT(TM("INFO: Skip uP7 file {%s}\n"), i_pPath);
            l_bAdd = false;
        }
    }

    if (l_bAdd)
    {
        m_cFiles.Push_Last(new CpreFile(this, i_pPath, m_eError));
    }

    return m_eError;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
eErrorCodes CpreManager::AddFile(const tXCHAR *i_pPath, Cfg::INode *i_pNode)
{
    if (i_pNode)
    {
        m_cFiles.Push_Last(new CpreFile(this, i_pPath, i_pNode, m_eError));
    }

    return m_eError;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
tBOOL CpreManager::CheckFileHash(const tXCHAR *i_pName, const tXCHAR *i_pHash)
{
    const size_t l_szHashSize = CKeccak::EBITS_256 / 8;
    tUINT8       l_pHash[l_szHashSize];
    size_t       l_szHashLen  = PStrLen(i_pHash);
    CWString     l_cPath(m_pSrcDir);

    l_cPath.Append(2, TM("/"), i_pName);

    tXCHAR *l_pPath = NormalizePath(l_cPath.Get());

    if (!l_pPath)
    {
        printf("Error: memory allocation error\n");
        return FALSE;
    }

    if (l_szHashLen != l_szHashSize * 2)
    {
        printf("Error: wrong hash size\n");
        return FALSE;
    }

    for (size_t l_szI = 0; l_szI < l_szHashSize; l_szI++)
    {
        CWString::Str2Hex(i_pHash + l_szI * 2, 2, l_pHash[l_szI]);
    }

    pAList_Cell l_pEl = NULL;

    while ((l_pEl = m_cFiles.Get_Next(l_pEl)))
    {
        CpreFile *l_pFile = m_cFiles.Get_Data(l_pEl);

        const tXCHAR *l_pFilePath  = l_pFile->GetOsPath();
        if (    (0 == PStrICmp(l_pFilePath, l_pPath))
             && (0 == memcmp(l_pFile->GetHash(), l_pHash, l_szHashSize))
           )
        {
            //OSPRINT(TM("INFO: Set file read-only, because HASH wasn't changed since last time {%s}\n"), i_pName);
            l_pFile->SetReadOnly(TRUE);
        }
    }

    free(l_pPath);
    return TRUE;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
tBOOL CpreManager::SetReadOnlyFile(const tXCHAR *i_pName)
{
    tXCHAR *l_pReadOnlyPath = NormalizePath(i_pName);
    if (!l_pReadOnlyPath)
    {
        printf("Error: memory allocation error\n");
        return FALSE;
    }

    pAList_Cell l_pEl            = NULL;
    size_t      l_szReadOnlyPath = PStrLen(l_pReadOnlyPath);

    while ((l_pEl = m_cFiles.Get_Next(l_pEl)))
    {
        CpreFile *l_pFile = m_cFiles.Get_Data(l_pEl);

        const tXCHAR *l_pFilePath  = l_pFile->GetOsPath();
        size_t        l_szFilePath = PStrLen(l_pFilePath);
        if (    (l_szFilePath >= l_szReadOnlyPath)
             && (0 == PStrICmp(l_pFilePath + l_szFilePath - l_szReadOnlyPath, l_pReadOnlyPath))
           )
        {
            l_pFile->SetReadOnly(TRUE);
        }
    }

    free(l_pReadOnlyPath);
    return TRUE;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
tBOOL CpreManager::SetExcludedFile(const tXCHAR *i_pName)
{
    tXCHAR *l_pReadOnlyPath = NormalizePath(i_pName);
    if (!l_pReadOnlyPath)
    {
        printf("Error: memory allocation error\n");
        return FALSE;
    }

    pAList_Cell l_pEl            = NULL;
    size_t      l_szReadOnlyPath = PStrLen(l_pReadOnlyPath);

    while ((l_pEl = m_cFiles.Get_Next(l_pEl)))
    {
        CpreFile *l_pFile = m_cFiles.Get_Data(l_pEl);

        const tXCHAR *l_pFilePath  = l_pFile->GetOsPath();
        size_t        l_szFilePath = PStrLen(l_pFilePath);
        if (    (l_szFilePath >= l_szReadOnlyPath)
             && (0 == PStrICmp(l_pFilePath + l_szFilePath - l_szReadOnlyPath, l_pReadOnlyPath))
           )
        {
            m_cFiles.Del(l_pEl, TRUE);
            break;
        }
    }

    free(l_pReadOnlyPath);
    return TRUE;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
int CpreManager::Process()
{
    struct stTraceId
    {
        CFuncRoot *pFunc;
        bool       bUsed;
    };

    pAList_Cell l_pFileEl     = NULL;
    eErrorCodes l_eError      = eErrorNo;
    size_t      l_szDir       = PStrLen(m_pSrcDir);
    tUINT64     l_qwEpochTime = GetEpochTime();
    tUINT32     l_uSessionId  = (tUINT32)((l_qwEpochTime - TIME_OFFSET_1601_1970) / TIME_SEC_100NS);
    tUINT16     l_wModuleId   = 0;
    tUINT16     l_wCounterId  = 0;
    stTraceId  *l_pTraceIdMap = NULL;
    size_t      l_szTraceIt   = 0;
    tBOOL       l_bNewSession = FALSE;

    while ((l_pFileEl = m_cFiles.Get_Next(l_pFileEl)))
    {
        CpreFile *l_pFile = m_cFiles.Get_Data(l_pFileEl);
        if (    (l_pFile)
             && (eErrorNo != (l_eError = l_pFile->GetError()))
           )
        {
            l_pFile->PrintErrors();
        }
    }

    if (eErrorNo != l_eError)
    {
        return l_eError;
    }


    //put read-only files at list head to manage trace ID conflicts
    l_pFileEl = m_cFiles.Get_First();
    while (l_pFileEl)
    {
        CpreFile *l_pFile = m_cFiles.Get_Data(l_pFileEl);
        if (    (l_pFile)
             && (l_pFile->GetReadOnly())
           )
        {
            pAList_Cell l_pNextEl = m_cFiles.Get_Next(l_pFileEl);
            m_cFiles.Del(l_pFileEl, FALSE);
            m_cFiles.Push_First(l_pFile);
            l_pFileEl = l_pNextEl;
        }
        else
        {
            l_pFileEl = m_cFiles.Get_Next(l_pFileEl);
        }
    }

    //create traces ID table to store information about used ID
    l_pTraceIdMap = (stTraceId *)malloc(MAXUINT16 * sizeof(stTraceId));
    if (!l_pTraceIdMap)
    {
        printf("ERROR: can't allocate memory\n");
        return eErrorMemAlloc;
    }

    memset(l_pTraceIdMap, 0, MAXUINT16 * sizeof(stTraceId));


    //generate IDs & save binary data
    l_pFileEl = NULL;
    while ((l_pFileEl = m_cFiles.Get_Next(l_pFileEl)))
    {
        CpreFile *l_pFile = m_cFiles.Get_Data(l_pFileEl);
        if (l_pFile)
        {
            CBList<CFuncRoot*> &l_rFunctions = l_pFile->GetFunctions();
            tBOOL               l_bReadOnly  = l_pFile->GetReadOnly();

            pAList_Cell l_pFuncEl    = NULL;
            while ((l_pFuncEl = l_rFunctions.Get_Next(l_pFuncEl)))
            {
                CFuncRoot *l_pFunc = l_rFunctions.Get_Data(l_pFuncEl);

                if (stFuncDesc::eType::eTrace == l_pFunc->GetType())
                {
                    CFuncTrace *l_pTrace = static_cast<CFuncTrace*>(l_pFunc);
                    
                               
                    stTraceId *l_pTraceId = &l_pTraceIdMap[l_pTrace->GetId()];
                    if (!l_pTraceId->bUsed)
                    {
                        l_pTraceId->bUsed = true;
                        l_pTraceId->pFunc = l_pFunc;
                    }
                    else
                    {
                        if (!l_bReadOnly)
                        {
                            while (l_szTraceIt < MAXUINT16)
                            {
                                l_pTraceId = &l_pTraceIdMap[l_szTraceIt];
                                if (l_pTraceId->bUsed)
                                {
                                    l_szTraceIt++;
                                }
                                else
                                {
                                    l_pTraceId->bUsed = true;
                                    l_pTraceId->pFunc = l_pFunc;
                                    l_pTrace->SetId((tUINT16)l_szTraceIt);
                                    OSPRINT(TM("INFO: Update trace ID at file {%s}:%d\n"), 
                                            l_pFile->GetOsPath() + l_szDir, 
                                            l_pFunc->GetLine());

                                    l_szTraceIt++;
                                    break;
                                }
                            }

                            if (l_szTraceIt >= MAXUINT16)
                            {
                                l_eError = eErrorTraceIdOverFlow;

                                OSPRINT(TM("ERROR: at file {%s}\n"), l_pFile->GetOsPath() + l_szDir);
                                printf(" * Line:%d, %s\n", l_pFunc->GetLine(), g_pErrorsText[l_eError]);
                            }
                        }
                        else
                        {
                            l_eError = eErrorTraceIdDuplicate;
                            OSPRINT(TM("ERROR: at file {%s}\n"), l_pFile->GetOsPath() + l_szDir);
                            printf(" * Line:%d, %s\n", l_pFunc->GetLine(), g_pErrorsText[l_eError]);
                            OSPRINT(TM(" * Conflicted with {%s}:%d\n"), 
                                    l_pTraceId->pFunc->GetFile()->GetOsPath() + l_szDir, 
                                    l_pTraceId->pFunc->GetLine());
                        }
                    }
                }
                else if (stFuncDesc::eType::eCreateCounter == l_pFunc->GetType())
                {
                    CFuncCounter *l_pCounter = static_cast<CFuncCounter*>(l_pFunc);
                    l_pCounter->SetId(l_wCounterId++);
                }
                else if (stFuncDesc::eType::eRegisterModule == l_pFunc->GetType())
                {
                    CFuncModule *l_pModule = static_cast<CFuncModule*>(l_pFunc);
                    l_pModule->SetId(l_wModuleId++);
                }
            }
        }
    }

    if (eErrorNo == l_eError)
    {
        l_pFileEl = NULL;
        while (    ((l_pFileEl = m_cFiles.Get_Next(l_pFileEl)))
                && (eErrorNo == l_eError)
              )
        {
            CpreFile *l_pFile = m_cFiles.Get_Data(l_pFileEl);
            if (l_pFile)
            {
                l_pFile->Save();
            }
        }

        l_eError = CreateSessionFile(l_uSessionId, l_bNewSession);
    }

    if (eErrorNo == l_eError)
    {
        l_eError = CreateHeaderFile(l_uSessionId, l_qwEpochTime, l_bNewSession);
    }

    free(l_pTraceIdMap);
    l_pTraceIdMap = NULL;

    return (int)l_eError;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
tBOOL CpreManager::SaveHashes(Cfg::INode *i_pFiles)
{
    const size_t  l_szHashSize = CKeccak::EBITS_256 / 8;
    pAList_Cell   l_pFileEl = NULL;
    tXCHAR        l_pTxtHash[l_szHashSize * 2 + 1];
    size_t        l_szDir   = PStrLen(m_pSrcDir);
    tBOOL         l_bReturn = TRUE;

    //sorting to do not change XML all the time
    while ((l_pFileEl = m_cFiles.Get_Next(l_pFileEl)))
    {
        pAList_Cell l_pMax = l_pFileEl;
        pAList_Cell l_pCur = l_pFileEl;

        while ((l_pCur = m_cFiles.Get_Next(l_pCur)))
        {
            CpreFile *l_pFileM = m_cFiles.Get_Data(l_pMax);
            CpreFile *l_pFileI = m_cFiles.Get_Data(l_pCur);

            if (0 > PStrICmp(l_pFileI->GetOsPath() + l_szDir, l_pFileM->GetOsPath() + l_szDir))
            {
                l_pMax = l_pCur;
            }
        }

        if (l_pMax != l_pFileEl)
        {
            m_cFiles.Extract(l_pMax);
            m_cFiles.Put_After(m_cFiles.Get_Prev(l_pFileEl), l_pMax);
            l_pFileEl = l_pMax;
        }
    } //while ((l_pFileEl = m_cFiles.Get_Next(l_pFileEl)))


    l_pFileEl = NULL;


    while ((l_pFileEl = m_cFiles.Get_Next(l_pFileEl)))
    {
        CpreFile *l_pFile = m_cFiles.Get_Data(l_pFileEl);
        if (    (l_pFile)
             && (!l_pFile->IsVirtual())
           )
        {
            const tUINT8 *l_pHash = l_pFile->GetHash();

            for (size_t l_szI = 0; l_szI < l_szHashSize; l_szI++)
            {
                PSPrint(l_pTxtHash + l_szI * 2, 3, TM("%02X"), l_pHash[l_szI]);
            }

            Cfg::INode *l_pFileNode = NULL;

            i_pFiles->AddChildEmpty(XML_NODE_FILES_ITEM, &l_pFileNode);
            if (l_pFileNode)
            {
                l_pFileNode->SetAttrText(XML_NARG_FILES_ITEM_RELATIVE_PATH, l_pFile->GetOsPath() + l_szDir);
                l_pFileNode->SetAttrText(XML_NARG_FILES_ITEM_HASH, l_pTxtHash);
                l_pFileNode->Release();
            }
            else
            {
                l_bReturn = FALSE;
                break;
            }
        }
    }

    return l_bReturn;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
eErrorCodes CpreManager::CreateSessionFile(tUINT32 i_uSession, tBOOL &o_rUpdated)
{
    //count if seconds since 01.01.1970
    CPFile      l_cuP7Binfile;
    eErrorCodes l_eError      = eErrorNo;
    CWString    l_cBinFilePath(m_pOutDir);
    size_t      l_szOldFile   = 0;
    tUINT8     *l_pOldFile    = NULL; 
    size_t      l_szNewFile   = 1024u*1024u;
    tUINT8     *l_pNewFile    = (tUINT8*)malloc(l_szNewFile);
    size_t      l_szNewData   = 0;
    tBOOL       l_bIsNew      = TRUE;

    o_rUpdated = FALSE;

    if (!l_pNewFile)
    {
        OSPRINT(TM("ERROR: Can't allocate memory %zu bytes\n"), l_szNewFile);
        return eErrorMemAlloc;
    }

    l_cBinFilePath.Append(3, TM("/"), m_pName, TM(".") uP7_FILES_EXT);

    if (CFSYS::File_Exists(l_cBinFilePath.Get()))
    {
        if (l_cuP7Binfile.Open(l_cBinFilePath.Get(), IFile::EOPEN | IFile::EACCESS_READ | IFile::ESHARE_READ))
        {
            l_szOldFile  = (size_t)l_cuP7Binfile.Get_Size();
            if (l_szOldFile)
            {
                l_pOldFile = (tUINT8*)malloc(l_szOldFile);
                if (l_pOldFile)
                {
                    if (l_szOldFile != l_cuP7Binfile.Read(l_pOldFile, l_szOldFile))
                    {
                        OSPRINT(TM("WARNING: Can't read file {%s}\n"), l_cBinFilePath.Get());
                        l_szOldFile = 0;
                    }
                }
                else
                {
                    OSPRINT(TM("WARNING: Can't allocate memory %zu\n"), l_szOldFile);
                    l_szOldFile = 0;
                }
            }

            l_cuP7Binfile.Close(FALSE);
        }
        else
        {
            OSPRINT(TM("WARNING: Can't open file {%s}\n"), l_cBinFilePath.Get());
        }
    }


    memcpy(l_pNewFile, &i_uSession, sizeof(i_uSession));
    l_szNewData += sizeof(i_uSession);


    //generate IDs & serialize binary data
    pAList_Cell l_pFileEl = NULL;
    while (    ((l_pFileEl = m_cFiles.Get_Next(l_pFileEl)))
            && (eErrorNo == l_eError)
          )
    {
        CpreFile *l_pFile = m_cFiles.Get_Data(l_pFileEl);
        if (l_pFile)
        {
            CBList<CFuncRoot*> &l_rFunctions = l_pFile->GetFunctions();

            pAList_Cell l_pFuncEl = NULL;
            while ((l_pFuncEl = l_rFunctions.Get_Next(l_pFuncEl)))
            {
                CFuncRoot *l_pFunc = l_rFunctions.Get_Data(l_pFuncEl);

                if (eErrorNo == l_eError)
                {
                    size_t l_szBuf = 0;
                    void  *l_pBuf  = l_pFunc->GetBuffer(l_szBuf);

                    if ((l_szBuf + l_szNewData) >= l_szNewFile)
                    {
                        size_t  l_szTemp = l_szNewFile * 2;
                        tUINT8 *l_pTemp  = (tUINT8*)realloc(l_pNewFile, l_szTemp);
                        if (l_pTemp)
                        {
                            l_pNewFile  = l_pTemp;
                            l_szNewFile = l_szTemp;
                        }
                        else
                        {
                            OSPRINT(TM("ERROR: Can't allocate memory %zu bytes\n"), l_szTemp);
                            l_eError = eErrorMemAlloc;
                        }
                    }

                    if (eErrorNo == l_eError)
                    {
                        memcpy(l_pNewFile + l_szNewData, l_pBuf, l_szBuf);
                        l_szNewData += l_szBuf;
                    }
                }
            }
        }
    }

    //compare ....
    if (    (l_szOldFile)
         && (l_szOldFile == l_szNewData)
         && (l_szNewData > sizeof(i_uSession))
         && (0 == memcmp(l_pNewFile + sizeof(i_uSession), l_pOldFile +  + sizeof(i_uSession), l_szNewData - sizeof(i_uSession)))
       )
    {
        OSPRINT(TM("INFO: session file remains the same {%s}\n"), l_cBinFilePath.Get());
        l_bIsNew = FALSE;
    }

    if (    (eErrorNo == l_eError)
         && (l_bIsNew)
       )
    {
        if (l_cuP7Binfile.Open(l_cBinFilePath.Get(), IFile::ECREATE | IFile::ESHARE_READ | IFile::EACCESS_WRITE))
        {
            if (l_szNewData == l_cuP7Binfile.Write(l_pNewFile, l_szNewData, FALSE))
            {
                o_rUpdated = TRUE;
            }
            else
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cBinFilePath.Get());
                return eErrorFileWrite;
            }

            l_cuP7Binfile.Close(TRUE);
        }
        else
        {
            OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cBinFilePath.Get());
            l_eError = eErrorFileWrite;
        }
    }

    if (l_pOldFile)
    {
        free(l_pOldFile);
        l_pOldFile = NULL;
    }

    if (l_pNewFile)
    {
        free(l_pNewFile);
        l_pNewFile = NULL;
    }

    return l_eError;
}


static const char *g_pVargs[]
{
    "euP7_arg_unk"      ,  //P7TRACE_ARG_TYPE_UNK    = 0x00,
    "euP7_arg_int8"     ,  //P7TRACE_ARG_TYPE_CHAR/P7TRACE_ARG_TYPE_INT8 = 0x01,
    "euP7_arg_char16"   ,  //P7TRACE_ARG_TYPE_CHAR16  ,//(0x02)
    "euP7_arg_int16"    ,  //P7TRACE_ARG_TYPE_INT16   ,//(0x03)
    "euP7_arg_int32"    ,  //P7TRACE_ARG_TYPE_INT32   ,//(0x04)
    "euP7_arg_int64"    ,  //P7TRACE_ARG_TYPE_INT64   ,//(0x05)
    "euP7_arg_double"   ,  //P7TRACE_ARG_TYPE_DOUBLE  ,//(0x06)
    "euP7_arg_pvoid"    ,  //P7TRACE_ARG_TYPE_PVOID   ,//(0x07)
    "euP7_arg_str_utf16",  //P7TRACE_ARG_TYPE_USTR16  ,//(0x08) //un
    "euP7_arg_str_ansi" ,  //P7TRACE_ARG_TYPE_STRA    ,//(0x09) //An
    "euP7_arg_str_utf8" ,  //P7TRACE_ARG_TYPE_USTR8   ,//(0x0A) //un
    "euP7_arg_str_utf32",  //P7TRACE_ARG_TYPE_USTR32  ,//(0x0B) //un
    "euP7_arg_char32"   ,  //P7TRACE_ARG_TYPE_CHAR32  ,//(0x0C)
    "euP7_arg_intmax"      //P7TRACE_ARG_TYPE_INTMAX  ,//(0x0D)
};


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
eErrorCodes CpreManager::CreateHeaderFile(tUINT32 i_uSession, tUINT64 i_qwEpochTime, tBOOL i_bUpdate)
{
    CWString             l_cHdrFilePath(m_pOutDir);
    CBList<CFuncTrace*>  l_cTraces;
    CBList<CFuncModule*> l_cModules;
    CBList<CFuncCounter*>l_cCounters;
    CPFile               l_cuP7Hdrfile;
    char                *l_pBuffer   = NULL;
    size_t               l_szBuffer  = 16384;
    eErrorCodes          l_eError    = eErrorNo;
    size_t               l_szText    = 0;
    const char          *l_pPreamble = 
        "////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////\n"
        "//                                                     WARNING!                                                       //\n"
        "//                                       this header is automatically generated                                       //\n"
        "//                                                 DO NOT MODIFY IT                                                   //\n"
        "//                                           Generated: %04u.%02u.%02u %02u:%02u:%02u                                           //\n"
        "////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////\n"
        "#ifndef UP7_TARGET_CPU_H\n"
        "#define UP7_TARGET_CPU_H\n\n"
        "//seconds count from 01.01.1970\n"
        "uint32_t g_uSessionId = %u;\n";
    



    l_cHdrFilePath.Append(1, TM("/uP7preprocessed.h"));

    if (    (!i_bUpdate)
         && (CFSYS::File_Exists(l_cHdrFilePath.Get()))
       )
    {
        return eErrorNo;
    }

    if (!l_cuP7Hdrfile.Open(l_cHdrFilePath.Get(), IFile::ECREATE | IFile::ESHARE_READ | IFile::EACCESS_WRITE))
    {
        OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
        return eErrorFileWrite;
    }

    l_pBuffer = (char*)malloc(l_szBuffer);
    if (!l_pBuffer)
    {
        OSPRINT(TM("ERROR: Can't allocate memory %zu\n"), l_szBuffer);
        return eErrorMemAlloc;
    }


    if (eErrorNo == l_eError)
    {
        tUINT32 l_uYear         = 0;
        tUINT32 l_uMonth        = 0;
        tUINT32 l_uDay          = 0;
        tUINT32 l_uHour         = 0;
        tUINT32 l_uMinutes      = 0;
        tUINT32 l_uSeconds      = 0;
        tUINT32 l_uMilliseconds = 0;
        tUINT32 l_uMicroseconds = 0;
        tUINT32 l_uNanoseconds  = 0;

        UnpackLocalTime(i_qwEpochTime,
                        l_uYear,
                        l_uMonth,
                        l_uDay,
                        l_uHour,
                        l_uMinutes,
                        l_uSeconds,
                        l_uMilliseconds,
                        l_uMicroseconds,
                        l_uNanoseconds
                       );

        l_szText = (size_t)sprintf(l_pBuffer, 
                                   l_pPreamble, 
                                   l_uYear,
                                   l_uMonth,
                                   l_uDay,
                                   l_uHour,
                                   l_uMinutes,
                                   l_uSeconds,
                                   i_uSession
            );


        if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
        {
            OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
            l_eError = eErrorFileWrite;
        }
    }


    if (eErrorNo == l_eError)
    {
        //create lists of modules, traces, telemetry
        pAList_Cell l_pFileEl = NULL;

        while ((l_pFileEl = m_cFiles.Get_Next(l_pFileEl)))
        {
            CpreFile *l_pFile = m_cFiles.Get_Data(l_pFileEl);
            if (l_pFile)
            {
                CBList<CFuncRoot*> &l_rFunctions = l_pFile->GetFunctions();

                pAList_Cell l_pFuncEl = NULL;
                while ((l_pFuncEl = l_rFunctions.Get_Next(l_pFuncEl)))
                {
                    CFuncRoot *l_pFunc = l_rFunctions.Get_Data(l_pFuncEl);

                    if (stFuncDesc::eType::eTrace == l_pFunc->GetType())
                    {
                        CFuncTrace *l_pTrace = static_cast<CFuncTrace*>(l_pFunc);

                        pAList_Cell l_pTraceEl = NULL;
                        bool        l_bAdded   = false;
                        while ((l_pTraceEl = l_cTraces.Get_Next(l_pTraceEl)))
                        {
                            if (l_cTraces.Get_Data(l_pTraceEl)->GetId() >= l_pTrace->GetId())
                            {
                                l_bAdded   = true; 
                                l_pTraceEl = l_cTraces.Get_Prev(l_pTraceEl);
                                l_cTraces.Add_After(l_pTraceEl, l_pTrace);
                                break;
                            }
                        }

                        if (!l_bAdded)
                        {
                            l_cTraces.Add_After(l_cTraces.Get_Last(), l_pTrace);
                        }
                    }
                    if (stFuncDesc::eType::eRegisterModule == l_pFunc->GetType())
                    {
                        CFuncModule *l_pMod = static_cast<CFuncModule*>(l_pFunc);

                        pAList_Cell l_pModEl = NULL;
                        bool        l_bAdded   = false;
                        while ((l_pModEl = l_cModules.Get_Next(l_pModEl)))
                        {
                            if (l_cModules.Get_Data(l_pModEl)->GetHash() >= l_pMod->GetHash())
                            {
                                l_bAdded   = true; 
                                l_pModEl = l_cModules.Get_Prev(l_pModEl);
                                l_cModules.Add_After(l_pModEl, l_pMod);
                                break;
                            }
                        }

                        if (!l_bAdded)
                        {
                            l_cModules.Add_After(l_cModules.Get_Last(), l_pMod);
                        }
                    }
                    else if (stFuncDesc::eType::eCreateCounter == l_pFunc->GetType())
                    {
                        CFuncCounter *l_pCounter   = static_cast<CFuncCounter*>(l_pFunc);
                        pAList_Cell   l_pCounterEl = NULL;
                        bool          l_bAdded     = false;
                        while ((l_pCounterEl = l_cCounters.Get_Next(l_pCounterEl)))
                        {
                            if (l_cCounters.Get_Data(l_pCounterEl)->GetHash() >= l_pCounter->GetHash())
                            {
                                l_bAdded   = true; 
                                l_pCounterEl = l_cCounters.Get_Prev(l_pCounterEl);
                                l_cCounters.Add_After(l_pCounterEl, l_pCounter);
                                break;
                            }
                        }

                        if (!l_bAdded)
                        {
                            l_cCounters.Add_After(l_cCounters.Get_Last(), l_pCounter);
                        }
                    }
                }
            }
        }
    }


    if (eErrorNo == l_eError)
    {
        if (l_cModules.Count())
        {
            l_szText = (size_t)sprintf(l_pBuffer, 
                                       "\n\nsize_t g_szModules = %u;\n"
                                       "struct stuP7Module g_pModules[] = \n{", 
                                       l_cModules.Count());

            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
        else
        {
            l_szText = (size_t)sprintf(l_pBuffer, "\n\nsize_t g_szModules = 0;\nstruct stuP7Module *g_pModules = NULL;\n\n");

            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
    }

    if (    (eErrorNo == l_eError)
         && (l_cModules.Count())
       )
    {
        //generate modules description
        pAList_Cell l_pModEl = NULL;
        bool        l_bFirst  = true;

        while (    ((l_pModEl = l_cModules.Get_Next(l_pModEl)))
                && (eErrorNo == l_eError)
              )
        {
            CFuncModule *l_pModule = l_cModules.Get_Data(l_pModEl);
            if (l_pModule)
            {
                l_szText = (size_t)sprintf(l_pBuffer, 
                                            "%s\n    {\"%s\", euP7Level_Trace, %u, %u}", 
                                            l_bFirst ? "" : ",",
                                            l_pModule->GetName(), 
                                            l_pModule->GetId(), 
                                            l_pModule->GetHash());

                l_bFirst = false;

                if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
                {
                    OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                    l_eError = eErrorFileWrite;
                    break;
                }
            }
        }

        if (eErrorNo == l_eError)
        {
            l_szText = (size_t)sprintf(l_pBuffer, "\n};\n\n"); 
            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
    }

    if (eErrorNo == l_eError)
    {
        if (l_cCounters.Count())
        {
            l_szText = (size_t)sprintf(l_pBuffer, 
                                       "size_t g_szTelemetry = %u;\n"
                                       "struct stuP7telemetry g_pTelemetry[] = \n{",
                                       l_cCounters.Count()
                                      ); 
            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
        else
        {
            l_szText = (size_t)sprintf(l_pBuffer, "size_t g_szTelemetry = 0;\nstruct stuP7telemetry *g_pTelemetry = NULL;\n\n");

            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
    }


    if (    (eErrorNo == l_eError)
         && (l_cCounters.Count())
       )
    {
        //generate modules description
        pAList_Cell l_pCntEl = NULL;
        bool        l_bFirst  = true;

        while (    ((l_pCntEl = l_cCounters.Get_Next(l_pCntEl)))
                && (eErrorNo == l_eError)
              )
        {
            CFuncCounter *l_pCounter = l_cCounters.Get_Data(l_pCntEl);
            if (l_pCounter)
            {
                l_szText = (size_t)sprintf(l_pBuffer, 
                                            "%s\n    {\"%s\", %u, %s, %u}", 
                                            l_bFirst ? "" : ",",
                                            l_pCounter->GetName(), 
                                            l_pCounter->GetHash(), 
                                            l_pCounter->GetEnabled() ? "true" : "false",
                                            l_pCounter->GetId()
                    );

                l_bFirst = false;

                if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
                {
                    OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                    l_eError = eErrorFileWrite;
                    break;
                }
            }
        }

        if (eErrorNo == l_eError)
        {
            l_szText = (size_t)sprintf(l_pBuffer, "\n};\n\n"); 
            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
    }

    if (    (eErrorNo == l_eError)
         && (l_cTraces.Count())
       )
    {
        //generate traces description
        pAList_Cell l_pTraceEl = NULL;

        while (    ((l_pTraceEl = l_cTraces.Get_Next(l_pTraceEl)))
                && (eErrorNo == l_eError)
              )
        {
            CFuncTrace         *l_pTrace = l_cTraces.Get_Data(l_pTraceEl);
            const sP7Trace_Arg *l_pVa    = NULL;
            size_t              l_szVA   = l_pTrace->GetVA(l_pVa);

            if (l_szVA)
            {
                l_szText = (size_t)sprintf(l_pBuffer, 
                                           "static const struct stuP7arg g_pArgsId%d[] = { ", 
                                           l_pTrace->GetId()
                                           );

                if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
                {
                    OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                    l_eError = eErrorFileWrite;
                    break;
                }



                for (size_t l_szI = 0; l_szI < l_szVA; l_szI++)
                {
                    if (l_pVa[l_szI].bType < LENGTH(g_pVargs))
                    {
                        l_szText = (size_t)sprintf(l_pBuffer, 
                                                    "{(uint8_t)%s, %u}%s", 
                                                    g_pVargs[l_pVa[l_szI].bType], 
                                                    l_pVa[l_szI].bSize,
                                                    ((l_szI + 1) == l_szVA) ? " };\n" : ", " 
                                                    );

                        if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
                        {
                            OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                            l_eError = eErrorFileWrite;
                            break;
                        }

                    }
                    else
                    {
                        printf("ERROR: trace format error parsing {%s} at {%s}:%d\n", 
                                l_pTrace->GetFormat(),
                                l_pTrace->GetFilePath(),
                                l_pTrace->GetLine()
                                );
                        l_eError = eErrorTraceFormat;
                        break;
                    }
                }
            }
        }
    }

    if (eErrorNo == l_eError)
    {
        if (l_cTraces.Count())
        {
            l_szText = (size_t)sprintf(l_pBuffer, 
                                       "\nsize_t g_szTraces = %u;\n"
                                       "struct stuP7Trace g_pTraces[] = \n{",
                                       l_cTraces.Count()
                                      ); 
            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
        else
        {
            l_szText = (size_t)sprintf(l_pBuffer, "size_t g_szTraces = 0;\nstruct stuP7Trace *g_pTraces = NULL;\n");

            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
    }

    if (    (eErrorNo == l_eError)
         && (l_cTraces.Count())
       )
    {
        //generate modules description
        bool        l_bFirst  = true;

        //generate traces description
        pAList_Cell l_pTraceEl = NULL;

        while (    ((l_pTraceEl = l_cTraces.Get_Next(l_pTraceEl)))
                && (eErrorNo == l_eError)
              )
        {
            CFuncTrace         *l_pTrace = l_cTraces.Get_Data(l_pTraceEl);
            const sP7Trace_Arg *l_pVa    = NULL;
            size_t              l_szVA   = l_pTrace->GetVA(l_pVa);

            if (l_szVA)
            {
                l_szText = (size_t)sprintf(l_pBuffer, 
                                            "%s\n    {%u, sizeof(g_pArgsId%u)/sizeof(struct stuP7arg), g_pArgsId%u}", 
                                            l_bFirst ? "" : ",",
                                            l_pTrace->GetId(),
                                            l_pTrace->GetId(),
                                            l_pTrace->GetId()
                                            );
            }
            else
            {
                l_szText = (size_t)sprintf(l_pBuffer, 
                                            "%s\n    {%u, 0, NULL}", 
                                            l_bFirst ? "" : ",",
                                            l_pTrace->GetId()
                                            );
            }
            l_bFirst = false;

            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
                break;
            }
        }

        if (eErrorNo == l_eError)
        {
            l_szText = (size_t)sprintf(l_pBuffer, "\n};"); 
            if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
            {
                OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
                l_eError = eErrorFileWrite;
            }
        }
    }

    if (eErrorNo == l_eError)
    {
        l_szText = (size_t)sprintf(l_pBuffer, "\n#endif //UP7_TARGET_CPU_H"); 
        if (l_szText != l_cuP7Hdrfile.Write((const tUINT8*)l_pBuffer, l_szText, FALSE))
        {
            OSPRINT(TM("ERROR: Can't write file {%s}\n"), l_cHdrFilePath.Get());
            l_eError = eErrorFileWrite;
        }
    }

    l_cuP7Hdrfile.Close(TRUE);

    l_cTraces.Clear(FALSE);
    l_cModules.Clear(FALSE);
    l_cCounters.Clear(FALSE);

    free(l_pBuffer);
    l_pBuffer = NULL;

    return l_eError;
}
